#!/usr/bin/env python3

"""
meteomod.py
METEOMOD model of metre-sized meteoroid population(s).

Reference: Broz et al. (2023, Nature, submit.).

"""

__author__ = "Miroslav Broz (miroslav.broz@email.cz)"
__version__ = "Nov 17th 2023"

import os
import re
import numpy as np

def read_tab5(filename="tab5.tex"):
    """Read Tab. 5 from Broz et al. (2023)."""

    tab5 = {}

    f = open(filename, "r")

    i = 0
    for line in f.readlines():
#        print(line)
        if line[0:1] == '%':
            continue

        line = re.sub("!", "", line)
#        line = re.sub("\$[^\$]*\$", "", line)
        line = re.sub("\$\^H\$", "", line)

        l = line.split("&")
        if len(l) < 10:
            continue
        if l[9].strip() == '--':
            continue
        if l[9].strip() == '':
            continue

        num = l[0].replace(" ","")
        if len(num) == 0:
            continue
        fam = re.sub(" *$", "", l[1])
        fam = re.sub("^ *", "", fam)
        N_neo1 = float(re.sub("-.*", "", l[8]))
        N_neo2 = float(re.sub(".*-", "", l[8]))
        try:
            Phi1 = float(re.sub("-.*", "", l[9]))
            Phi2 = float(re.sub(".*-", "", l[9]))
        except ValueError:
            continue

        try:
            tau_neo = float(l[4].replace("?", ""))
            tau_mb = float(l[5].replace("?", ""))
            N_mb1 = float(re.sub("-.*", "", l[7]))
            N_mb2 = float(re.sub(".*-", "", l[7]))
        except ValueError as ex:
            tau_neo = 0.0
            tau_mb = 0.0
            N_mb1 = 0.0
            N_mb2 = 0.0
            pass

        tmp = { \
            'i': i,
            'num': num,
            'fam': fam,
            'tau_neo': tau_neo,
            'tau_mb': tau_mb,
            'N_mb1': N_mb1,
            'N_mb2': N_mb2,
            'N_neo1': N_neo1,
            'N_neo2': N_neo2,
            'Phi1': Phi1,
            'Phi2': Phi2,
            }
        i += 1

        tab5[num] = tmp
    
    f.close()

    return tab5

def read_map(filename="map.out"):
    """Read map of probability for one orbital simulation."""

    print("Reading map '", filename, "'.")
    map = np.zeros((350,90))

    f = open(filename, "r")

    for line in f.readlines():
        l = line.split()
        if len(l) < 4:
           continue
        i = int(l[0])-1
        j = int(l[1])-1
        val = float(l[4])

        map[i,j] = val

    f.close()

    map /= np.sum(map)
    return map

def read_maps(filename="maps.in", binary="maps.npz"):
    """Read all maps of probability."""

    # Note: Wtf. is the [()] thing? Get dict out of array!
    if os.path.isfile(binary):
        maps = np.load(binary, allow_pickle=True)
        maps = maps['arr_0'][()]
        return maps

    maps = {}

    f = open("maps.in", "r")
    for line in f.readlines():
        if line[0:1] == '%':
            continue
        l = line[:-1].split("&")
        if len(l) < 3:
            continue

        num = l[0].replace(" ", "")
        fam = re.sub(" *$", "", l[1])
        fam = re.sub("^ *", "", fam)
        fil = l[2].strip()

        maps[num] = read_map("map/%s/map.out" % fil)

    f.close()

    np.savez(binary, maps)

    return maps

def read_orbits(filename="orbits.in"):

    orbits = []

    f = open(filename, "r")
    for line in f.readlines():
        if line[0:1] == '#':
            continue
        l = line.split()
        if len(l) < 3:
            continue
        orbit = {'a': float(l[0]), 'e': float(l[1]), 'inc': float(l[2]), 'dsg': l[3]}
        orbits.append(orbit)
    f.close()

    return orbits

def compute_p(orbit, maps=None, tab5=None):
    """Compute probability of coming from a source, based on maps and N_neo populations."""

    a1 = 0.0
    a2 = 3.50
    da = 0.01

    i1 = 0.0
    i2 = 90.0
    di = 1.0

    a = orbit['a']
    inc = orbit['inc']
    i = int((a-a1)/da)
    j = int((inc-i1)/di)

#    print("i = ", i)
#    print("j = ", j)
#    print("maps = ", maps)

    vals = {}
    p = {}
    s = 0.0
    for key in maps.keys():
        map = maps[key]
        val = map[i,j]
        vals[key] = val

        tmp = tab5[key]
        N_neo = 0.5*(tmp['N_neo1'] + tmp['N_neo2'])
        val *= N_neo

        p[key] = val
        s += val

    for key in p.keys():
        p[key] /= s

    return p, vals

def sort_dict(p):
    """Sort dictionary according to values."""

    q = {}
    keys = list(p.keys())
    vals = list(p.values())
    idxs = np.argsort(vals)
    q = { keys[i]: vals[i] for i in idxs }

    return q

def main():
    """Run METEOMOD model."""

    tab5 = read_tab5()
    maps = read_maps()
    orbits = read_orbits("orbits.tmp")

    f = open("maps.tmp", "r")
    tmp = f.read()
    f.close()
    tmp = tmp.split()

    keys = list(maps.keys())
    for key in keys:
        if key not in tmp:
            maps.pop(key)

    f = open("meteomod.out", "w")
    f.write("# Reference: Broz et al. (2024, A&A 689, A183), Tab. 5.\n")

    for orbit in orbits:
        p, mapvals = compute_p(orbit, maps=maps, tab5=tab5)

        q = p
#        q = sort_dict(p)

        keys = list(q.keys())
#        keys.reverse()

        f.write("# dsg = %s\n" % orbit['dsg'])
        f.write("# a = %f au\n" % orbit['a'])
        f.write("# inc = %f deg\n" % orbit['inc'])
        f.write("# i  num  p               fam                     tau_neo  tau_mb     N_mb1    N_mb2   N_neo1   N_neo2   Phi1     Phi2          M_j\n")
        f.write("# -  -    1               -                       My       My         10^10    ditto   10^8     ditto    10^-9 km^-2 y^-1 ditto 1\n")

        i = 0
        for key in keys:
            i += 1
            f.write("%2d  %4s  %.8e  \"%16s\"	%8.2f %8.2f %8.2f %8.2f %8.2f %8.2f %8.2f %8.2f  %e\n" % (i, key, q[key], tab5[key]['fam'], tab5[key]['tau_neo'], tab5[key]['tau_mb'], tab5[key]['N_mb1'], tab5[key]['N_mb2'], tab5[key]['N_neo1'], tab5[key]['N_neo2'], tab5[key]['Phi1'], tab5[key]['Phi2'], mapvals[key]))

    f.close()

if __name__ == "__main__":
    main()


