function ent = wentropy(x,t_ent,in3)
%WENTROPY Entropy (wavelet packet).
%   E = WENTROPY(X,T,P) returns the entropy E of the
%   vector or matrix X. 
%   In both cases, output E is a real number.
%
%   T is a string containing the type of entropy:
%     T = 'shannon', 'threshold', 'norm',
%     'log energy' (or 'logenergy'), 'sure', 'user'.
%
%   P is an optional parameter depending on the value of T:
%     If T = 'shannon' or 'log energy', P is not used.
%     If T = 'threshold' or 'sure', P is the
%     threshold and must a positive number.
%     If T = 'norm', P is the power and must be such
%     that 1 <= P.
%     If T = 'user', P is a string containing
%     the M-file name of your own entropy
%     function, with a single input X.
%
%   E = WENTROPY(X,T) is equivalent to E = WENTROPY(X,T,0).

%   M. Misiti, Y. Misiti, G. Oppenheim, J.M. Poggi 12-Mar-96.
%   Last Revision: 02-Aug-2000.
%   Copyright 1995-2001 The MathWorks, Inc.
%   $Revision: 1.11 $

% Check arguments.
if errargn(mfilename,nargin,[2 3],nargout,[0 1]), error('*'); end

x = x(:);
switch t_ent
    case 'shannon'       % in3 not used.
      x = x(find(x)).^2;
      ent = -sum(x.*log(eps+x));

    case 'threshold'     % in3 is the threshold.
      if nargin==2 | isempty(in3) | ischar(in3) | in3<0 , errStop; end
      x = abs(x);
      ent = sum(x > in3);

    case 'sure'          % in3 is the threshold.
      if nargin==2 | isempty(in3) | ischar(in3) | in3<0 , errStop; end
      n = length(x);
      x2 = x.^2; t2 = in3.^2;
      xgt =  sum(x2 > t2); xlt = n - xgt;
      ent = n - (2*xlt) + (t2*xgt) + sum(x2.*(x2 <= t2));

    case 'norm'          % in3 = p , ent = (lp_norm)^p.
      if nargin==2 | isempty(in3) | ischar(in3) | in3<1 , errStop; end
      x = abs(x);
      ent = sum(x.^in3);

    case {'energy','log energy','logenergy'}     % in3 not used.
      x = x(find(x)).^2;
      ent = sum(log(x));

    case 'user'  % in3 = '<function>' , user entropy.
      if nargin==2 | isempty(in3) | ~ischar(in3) , errStop; end
      ent = feval(in3,x);

    otherwise
      errStop;        
end

prec = 1.0E-10;
if abs(ent)<prec , ent = 0; end

% Internal Function
function errStop
  errargt(mfilename,'invalid argument value','msg');
  error('*');  
