function [c,l] = wavedec(x,n,varargin)
%WAVEDEC Multi-level 1-D wavelet decomposition.
%   WAVEDEC performs a multilevel 1-D wavelet analysis
%   using either a specific wavelet 'wname' or a specific set 
%   of wavelet decomposition filters (see WFILTERS).
%
%   [C,L] = WAVEDEC(X,N,'wname') returns the wavelet
%   decomposition of the signal X at level N, using 'wname'.
%
%   N must be a strictly positive integer (see WMAXLEV).
%   The output decomposition structure contains the wavelet
%   decomposition vector C and the bookkeeping vector L.
%
%   For [C,L] = WAVEDEC(X,N,Lo_D,Hi_D),
%   Lo_D is the decomposition low-pass filter and
%   Hi_D is the decomposition high-pass filter.
%
%   The structure is organized as:
%   C      = [app. coef.(N)|det. coef.(N)|... |det. coef.(1)]
%   L(1)   = length of app. coef.(N)
%   L(i)   = length of det. coef.(N-i+2) for i = 2,...,N+1
%   L(N+2) = length(X).
%
%   See also DWT, WAVEINFO, WFILTERS, WMAXLEV.

%   M. Misiti, Y. Misiti, G. Oppenheim, J.M. Poggi 12-Mar-96.
%   Last Revision: 16-Jan-2001.
%   Copyright 1995-2001 The MathWorks, Inc.
%   $Revision: 1.13 $ $Date: 2001/03/30 16:04:10 $

% Check arguments.
if errargn(mfilename,nargin,[3:4],nargout,[0:2]), error('*'), end
if errargt(mfilename,n,'int'), error('*'), end
if nargin==3
    [Lo_D,Hi_D] = wfilters(varargin{1},'d');
else
    Lo_D = varargin{1};   Hi_D = varargin{2};
end

% Initialization.
s = size(x); x = x(:)'; % row vector
c = [];      l = [length(x)];

for k = 1:n
    [x,d] = dwt(x,Lo_D,Hi_D); % decomposition
    c     = [d c];            % store detail
    l     = [length(d) l];    % store length
end

% Last approximation.
c = [x c];
l = [length(x) l];

if s(1)>1, c = c'; l = l'; end
