function [psi,X] = shanwavf(varargin)
%SHANWAVF Complex Shannon wavelet.
%   [PSI,X] = SHANWAVF(LB,UB,N,FB,FC) returns values of
%   the complex Shannon wavelet defined by a bandwidth parameter,
%   FB, a wavelet center frequency, FC.
%   FB and FC must be such that FC > FB/2 > 0 .
%
%   The function PSI is computed using the explicit expression:
%   PSI(X) = (FB^0.5)*(sinc(FB*X).*exp(2*i*pi*FC*X))
%   on an N point regular grid in the interval [LB,UB].
%
%   Output arguments are the wavelet function PSI
%   computed on the grid X.
%
%   See also WAVEINFO.

%   M. Misiti, Y. Misiti, G. Oppenheim, J.M. Poggi 09-Jun-99.
%   Last Revision: 02-Aug-2000.
%   Copyright 1995-2001 The MathWorks, Inc.
%   $Revision: 1.5 $  $Date: 2001/03/30 16:03:47 $

% Check arguments.
if errargn(mfilename,nargin,[4 5],nargout,[0:2]), error('*'); end

X = linspace(varargin{1:3});  % wavelet support.
Fc = 1;
Fb = 1;
switch nargin
  case 4
    if ischar(varargin{4})
       label = deblank(varargin{4});
       ind   = strncmpi('shan',label,4);
       if isequal(ind,1)
          label([1:4]) = [];
          len = length(label);
          if len>0
             ind = findstr('-',label);
             if isempty(ind)
                Fb = []; % error 
             else
                Fc = wstr2num(label(1:ind-1));
                label(1:ind) = [];
                Fb = wstr2num(label);    
             end
          else
            Fc = []; % error     
          end
       else
          Fc = []; % error 
       end
    else
      Fc = []; % error 
    end
  case 5 , Fc = varargin{4}; Fb = varargin{5};
end
if isempty(Fc) | isempty(Fb) , err = 1; else , err = 0; end
if ~err 
    err = ~isnumeric(Fc) | ~isnumeric(Fb) | (Fc<=0) | (Fb<=0) | (Fc<=Fb/2);
end
if err
    msg = 'Invalid Wavelet Number!';
    error(msg)
end
psi = (Fb^0.5)*(sinc(Fb*X).*exp(2*i*pi*Fc*X));



function y = sinc(x)
%
%               | sin(pi*x)/(pi*x)  if x ~= 0
% y = sinc(x) = |
%               | 1                 if x == 0

y = ones(size(x));
k = find(x);
y(k) = sin(pi*x(k))./(pi*x(k));
