function varargout = meyer(varargin);
%MEYER Meyer wavelet.
%   [PHI,PSI,T] = MEYER(LB,UB,N) returns Meyer
%   scaling and wavelet functions evaluated on
%   an N point regular grid in the interval [LB,UB].
%   N must be a power of two.
%
%   Output arguments are the scaling function PHI, the
%   wavelet function PSI computed on the grid T.
%   These functions have [-8 8] as effective support.
%
%   A fourth argument is allowed, if only one function
%   is required:
%     [PHI,T] = MEYER(LB,UB,N,'phi')
%     [PSI,T] = MEYER(LB,UB,N,'psi')
%   When the fourth argument is used, but not equal to
%   'phi' or 'psi', outputs are the same as in main option.
%
%   See also MEYERAUX, WAVEFUN, WAVEINFO.

%   M. Misiti, Y. Misiti, G. Oppenheim, J.M. Poggi 12-Mar-96.
%   Last Revision: 16-Jan-2001.
%   Copyright 1995-2001 The MathWorks, Inc.
%   $Revision: 1.12 $ $Date: 2001/03/30 16:03:31 $

% Check arguments.
if errargn(mfilename,nargin,[3:4],nargout,[0:3]), error('*'), end
switch nargin
  case 3
    [lowb,uppb,n] = deal(varargin{:}); opt = 'two';

  case 4
    [lowb,uppb,n,opt] = deal(varargin{:});
    if ~(isequal(opt,'two') | isequal(opt,'phi') | isequal(opt,'psi'))
        opt = 'two';
    end
end
tmp = log(n)/log(2);
if tmp ~= fix(tmp)
	msg = ['n must be a power of two !'];
	errargt(mfilename,msg,'msg',0);
    error('*')
end
tmp = uppb-lowb;
if tmp<0
	msg = ['The interval [LB,UB] is not valid !'];
	errargt(mfilename,msg,'msg',0);
    error('*')
end

% Transform interval bounds to grid.
lint = (uppb-lowb)/2/pi; 
x    = [-n:2:n-2]/(2*lint);
xa   = abs(x);

% Scaling function phi.
if opt == 'phi' | opt == 'two'

    % Compute support of Fourier transform of phi.
    int1 = find((xa < 2*pi/3));
    int2 = find((xa >= 2*pi/3) & (xa < 4*pi/3));

    % Compute Fourier transform of phi.
    phihat = zeros(1,n);
    phihat(int1) = ones(size(int1));
    phihat(int2) = cos(pi/2*meyeraux(3/2/pi*xa(int2)-1));

    % Compute phi using non standard inverse FFT.
    [phi,t] = instdfft(phihat,lowb,uppb);
end

% Wavelet function psi.
if opt == 'psi' | opt == 'two'

    % Compute support of Fourier transform of psi.
    int1 = find((xa >= 2*pi/3) & (xa < 4*pi/3)); 
    int2 = find((xa >= 4*pi/3) & (xa < 8*pi/3));

    % Compute Fourier transform of psi.
    psihat = zeros(1,n);
    psihat(int1) = exp(i*x(int1)/2).*sin(pi/2*meyeraux(3/2/pi*xa(int1)-1));
    psihat(int2) = exp(i*x(int2)/2).*cos(pi/2*meyeraux(3/4/pi*xa(int2)-1));

    % Compute psi using non standard inverse FFT.
    [psi,t] = instdfft(psihat,lowb,uppb);
end

% Set output arguments.
switch opt
    case 'psi' , varargout = {psi,t};
    case 'phi' , varargout = {phi,t};
    otherwise  , varargout = {phi,psi,t};
end
