function x = idwt2(a,h,v,d,varargin)
%IDWT2  Single-level inverse discrete 2-D wavelet transform.
%   IDWT2 performs a single-level 2-D wavelet reconstruction
%   with respect to either a particular wavelet
%   ('wname', see WFILTERS for more information) or particular wavelet
%   reconstruction filters (Lo_R and Hi_R) you specify.
%
%   X = IDWT2(CA,CH,CV,CD,'wname') uses the wavelet 'wname'
%   to compute the single-level reconstructed approximation
%   coefficients matrix X, based on approximation matrix CA
%   and (horizontal, vertical, and diagonal) details matrices
%   CH, CV and CD.
%
%   X = IDWT2(CA,CH,CV,CD,Lo_R,Hi_R) reconstructs as above,
%   using filters you specify:
%   Lo_R is the reconstruction low-pass filter.
%   Hi_R is the reconstruction high-pass filter.
%   Lo_R and Hi_R must be the same length.
%
%   Let SA = size(CA) = size(CH) = size(CV) = size(CD) and
%   LF the length of the filters; then size(X) = SX where
%   SX = 2*SA if the DWT extension mode is set to periodization.
%   SX = 2*SA-LF+2 for the other extension modes.
%
%   X = IDWT2(CA,CH,CV,CD,'wname',S) and
%   X = IDWT2(CA,CH,CV,CD,Lo_R,Hi_R,S) return the size-S
%   central portion of the result obtained using
%   IDWT2(CA,CH,CV,CD,'wname'). S must be less than SX.
%
%   X = IDWT2(...,'mode',MODE) computes the wavelet
%   reconstruction using the specified extension mode MODE.
%
%   X = IDWT2(CA,[],[],[], ... ) returns the single-level
%   reconstructed approximation coefficients matrix X
%   based on approximation coefficients matrix CA.
%   
%   X = IDWT2([],CH,[],[], ... ) returns the single-level
%   reconstructed detail coefficients matrix X
%   based on horizontal detail coefficients matrix CH.
%
%   The same result holds for X = IDWT2([],[],CV,[], ... ) and
%   X = IDWT2([],[],[],CD, ... ).
%
%   More generally, X = IDWT2(AA,HH,VV,DD, ... ) returns the single-level
%   reconstructed matrix X where AA can be CA or [], and so on.
%
%   See also DWT2, DWTMODE, UPWLEV2.

%   M. Misiti, Y. Misiti, G. Oppenheim, J.M. Poggi 12-Mar-96.
%   Last Revision: 02-Aug-2000.
%   Copyright 1995-2001 The MathWorks, Inc.
% $Revision: 1.14 $

% Check arguments.
if errargn(mfilename,nargin,[5:11],nargout,[0 1]), error('*'), end
if isempty(a) & isempty(h) & isempty(v) & isempty(d), x = []; return; end

if ischar(varargin{1})
    [Lo_R,Hi_R] = wfilters(varargin{1},'r'); next = 2;
else
    Lo_R = varargin{1}; Hi_R = varargin{2};  next = 3;
end

% Default: Size, Shift and Extension.
sx      = [];
dwtATTR = dwtmode('get');

% Check arguments for Size, Shift and Extension.
k = next;
while k<=length(varargin)
    if ischar(varargin{k})
        switch varargin{k}
           case 'mode'  , dwtATTR.extMode = varargin{k+1};
           case 'shift' , dwtATTR.shift2D = mod(varargin{k+1},2);
        end
        k = k+2;
    else
        sx = varargin{k}; k = k+1;
    end
end
      
x = upsaconv('2D',a,{Lo_R,Lo_R},sx,dwtATTR)+ ... % Approximation.
    upsaconv('2D',h,{Hi_R,Lo_R},sx,dwtATTR)+ ... % Horizontal Detail.
    upsaconv('2D',v,{Lo_R,Hi_R},sx,dwtATTR)+ ... % Vertical Detail.
    upsaconv('2D',d,{Hi_R,Hi_R},sx,dwtATTR);     % Diagonal Detail.
