function [a,h,v,d] = dwt2(x,varargin)
%DWT2 Single-level discrete 2-D wavelet transform.
%   DWT2 performs a single-level 2-D wavelet decomposition
%   with respect to either a particular wavelet ('wname',
%   see WFILTERS for more information) or particular wavelet filters
%   (Lo_D and Hi_D) you specify.
%
%   [CA,CH,CV,CD] = DWT2(X,'wname') computes the approximation
%   coefficients matrix CA and details coefficients matrices 
%   CH, CV, CD, obtained by a wavelet decomposition of the 
%   input matrix X.
%   'wname' is a string containing the wavelet name.
%
%   [CA,CH,CV,CD] = DWT2(X,Lo_D,Hi_D) computes the 2-D wavelet
%   decomposition as above given these filters as input:
%   Lo_D is the decomposition low-pass filter.
%   Hi_D is the decomposition high-pass filter.
%   Lo_D and Hi_D must be the same length.
%
%   Let SX = size(X) and LF = the length of filters; then
%   size(CA) = size(CH) = size(CV) = size(CD) = SA where
%   SA = CEIL(SX/2), if the DWT extension mode is set to
%   periodization. SA = FLOOR((SX+LF-1)/2) for the other
%   extension modes. For the different DWT extension modes, 
%   see DWTMODE.
%
%   [CA,CH,CV,CD] = DWT2(...,'mode',MODE) computes the wavelet 
%   decomposition with the extension mode MODE you specify.
%   MODE is a string containing the extension mode.
%   Example: 
%     [ca,ch,cv,cd] = dwt2(x,'db1','mode','sym');
%
%   See also DWTMODE, IDWT2, WAVEDEC2, WAVEINFO.

%   M. Misiti, Y. Misiti, G. Oppenheim, J.M. Poggi 12-Mar-96.
%   Last Revision: 02-Aug-2000.
%   Copyright 1995-2001 The MathWorks, Inc.
% $Revision: 1.14 $

% Check arguments.
if errargn(mfilename,nargin,[2:7],nargout,[0,1,4]), error('*'), end
if ischar(varargin{1})
    [Lo_D,Hi_D] = wfilters(varargin{1},'d'); next = 2;
else
    Lo_D = varargin{1}; Hi_D = varargin{2};  next = 3;
end

% Default: Shift and Extension.
dwtATTR = dwtmode('get');
shift   = dwtATTR.shift2D;
dwtEXTM = dwtATTR.extMode;

% Check arguments for Extension and Shift.
for k = next:2:nargin-1
    switch varargin{k}
      case 'mode'  , dwtEXTM = varargin{k+1};
      case 'shift' , shift   = mod(varargin{k+1},2);
    end
end

% Compute sizes.
lf = length(Lo_D);
sx = size(x);

% Extend, Decompose &  Extract coefficients.
flagPer = isequal(dwtEXTM,'per');
if ~flagPer
    sizeEXT = lf-1; sizeKEPT = sx+lf-1;
else
    sizeEXT = lf/2; sizeKEPT = 2*ceil(sx/2);
end
y = wextend('2D',dwtEXTM,x,[sizeEXT,sizeEXT]);
z = wconv('row',y,Lo_D);
a = convdown(z,Lo_D,sizeKEPT,shift);
h = convdown(z,Hi_D,sizeKEPT,shift);
z = wconv('row',y,Hi_D);
v = convdown(z,Lo_D,sizeKEPT,shift);
d = convdown(z,Hi_D,sizeKEPT,shift);

%-------------------------------------------------------%
% Internal Function(s)
%-------------------------------------------------------%
function y = convdown(x,f,sizeKEPT,shift)

y = wconv('col',x,f);
y = wkeep(y,sizeKEPT);
y = dyaddown(y,'col',shift(2));
y = dyaddown(y,'row',shift(1));
%-------------------------------------------------------%

