function [a,d] = dwt(x,varargin)
%DWT Single-level discrete 1-D wavelet transform.
%   DWT performs a single-level 1-D wavelet decomposition
%   with respect to either a particular wavelet ('wname',
%   see WFILTERS for more information) or particular wavelet filters
%   (Lo_D and Hi_D) that you specify.
%
%   [CA,CD] = DWT(X,'wname') computes the approximation
%   coefficients vector CA and detail coefficients vector CD,
%   obtained by a wavelet decomposition of the vector X.
%   'wname' is a string containing the wavelet name.
%
%   [CA,CD] = DWT(X,Lo_D,Hi_D) computes the wavelet decomposition
%   as above given these filters as input:
%   Lo_D is the decomposition low-pass filter.
%   Hi_D is the decomposition high-pass filter.
%   Lo_D and Hi_D must be the same length.
%
%   Let LX = length(X) and LF = the length of filters; then
%   length(CA) = length(CD) = LA where LA = CEIL(LX/2),
%   if the DWT extension mode is set to periodization.
%   LA = FLOOR((LX+LF-1)/2) for the other extension modes.  
%   For the different signal extension modes, see DWTMODE. 
%
%   [CA,CD] = DWT(...,'mode',MODE) computes the wavelet 
%   decomposition with the extension mode MODE you specify.
%   MODE is a string containing the extension mode.
%   Example: 
%     [ca,cd] = dwt(x,'db1','mode','sym');
%
%   See also DWTMODE, IDWT, WAVEDEC, WAVEINFO.

%   M. Misiti, Y. Misiti, G. Oppenheim, J.M. Poggi 12-Mar-96.
%   Last Revision: 02-Aug-2000.
%   Copyright 1995-2001 The MathWorks, Inc.
%   $Revision: 1.14 $

% Check arguments.
if errargn(mfilename,nargin,[2:7],nargout,[0:2]), error('*'), end

if ischar(varargin{1})
    [Lo_D,Hi_D] = wfilters(varargin{1},'d'); next = 2;
else
    Lo_D = varargin{1}; Hi_D = varargin{2};  next = 3;
end

% Default: Shift and Extension.
dwtATTR = dwtmode('get');
shift   = dwtATTR.shift1D;
dwtEXTM = dwtATTR.extMode;

% Check arguments for Extension and Shift.
for k = next:2:nargin-1
    switch varargin{k}
      case 'mode'  , dwtEXTM = varargin{k+1};
      case 'shift' , shift   = mod(varargin{k+1},2);
    end
end

% Compute sizes.
lf = length(Lo_D);
lx = length(x);

% Extend, Decompose &  Extract coefficients.
flagPer = isequal(dwtEXTM,'per');
if ~flagPer
    lenEXT = lf-1; lenKEPT = lx+lf-1;
else
    lenEXT = lf/2; lenKEPT = 2*ceil(lx/2);
end
y = wextend('1D',dwtEXTM,x,lenEXT);
a = convdown(y,Lo_D,lenKEPT,shift);
d = convdown(y,Hi_D,lenKEPT,shift);

%-----------------------------------------------------%
% Internal Function(s)
%-----------------------------------------------------%
function y = convdown(x,f,lenKEPT,shift)

y = wconv('1D',x,f);
y = wkeep(y,lenKEPT);
y = dyaddown(y,shift);
%-----------------------------------------------------%
