function varargout = wtreemgr(option,t,varargin)
%WTREEMGR NTREE object manager.
%   VARARGOUT = WTREEMGR(OPTION,T,VARARGIN)
%   Allowed values for OPTION and associated uses are
%   described in the functions listed in the See also section:
%
%   'order'    : Order of tree.
%   'depth'    : Depth of tree.
%   'leaves'   : Terminal nodes.
%   'tnodes'   : Terminal nodes.
%   'noleaves' : Not Terminal nodes.
%   'allnodes' : All nodes.
%   'isnode'   : Is node.
%   'istnode'  : Lop "is terminal node".
%   'nodeasc'  : Node ascendants.
%   'nodedesc' : Node descendants.
%   'nodepar'  : Node parent.
%   'ntnode'   : Number of terminal nodes.
%
%    See also ALLNODES, ISNODE, ISTNODE, LEAVES, NODEASC, NODEDESC,
%             NODEPAR, NOLEAVES, NTNODE, TNODES, TREEDPTH, TREEORD.

%   M. Misiti, Y. Misiti, G. Oppenheim, J.M. Poggi 01-Jan-97.
%   Last Revision: 16-Sep-1999.
%   Copyright 1995-2001 The MathWorks, Inc.
%   $Revision: 1.4 $  $Date: 2001/03/30 16:25:48 $

switch option
    case {'order','depth'} , varargout{1} = get(t,option);
    
    case 'leaves'    
        [order,tn] = get(t,'order','tn');
        K = [1:length(tn)]';
        if nargin>2
            flagdps = varargin{1};
            switch flagdps
                case {'s','sort'}
                  [tn,K] = sort(tn); [nul,K] = sort(K);

                case {'sdp','dps','sortdp','dpsort'}
                  [tn,K] = sort(tn); [nul,K] = sort(K);
                  [tn(:,1),tn(:,2)] = ind2depo(order,tn);

                case {'dp'}
                  [tn(:,1),tn(:,2)] = ind2depo(order,tn);
            end
        end
        varargout = {tn,K};
  
    case 'tnodes'
        if nargin==2 , outType = 'sort'; else , outType = 'sortdp'; end
        [varargout{1},varargout{2}] = leaves(t,outType);

    case 'noleaves'    
        if nargin==2 , flagdp = 0; else , flagdp = varargin{1}; end 
        varargout{1} = descendants(t,0,'not_tn',flagdp);

    case 'allnodes'
        [order,depth,varargout{1}] = get(t,'order','depth','tn');
        if (length(varargout{1})==1) & (depth==0) , return; end
        if nargin==2 , flagdp = 0; else , flagdp = 1; end
        varargout{1} = ascendants(varargout{1},order,depth,flagdp);
        
    case 'isnode'
        order = get(t,'order');
        node  = depo2ind(order,varargin{1});
        allN  = allnodes(t);
        if prod(size(node))<=1
            if find(allN==node), varargout{1} = 1;
            else , varargout{1} = 0;
            end
        else
            varargout{1} = ismember(node,allN);
        end

    case 'istnode'
        [order,tn] = get(t,'order','tn');
        node = depo2ind(order,varargin{1});
        varargout{1} = zeros(size(node));
        for k = 1:length(node)
            in = find(node(k)==tn);
            if ~isempty(in), varargout{1}(k) = in; end
        end

    case 'nodeasc'
        order = get(t,'order');
        node  = depo2ind(order,varargin{1});
        [d,p] = ind2depo(order,node);
        if nargin==3 , flagdp = 0; else , flagdp = 1; end
        varargout{1} = flipud(ascendants(node,order,d,flagdp));

    case 'nodedesc'
        if nargin==3 , flagdp = 0; else , flagdp = 1; end
        varargout{1} = descendants(t,varargin{1},'all',flagdp);

    case 'nodepar'
        order = get(t,'order');
        node  = depo2ind(order,varargin{1});
        par   = floor((node-1)/order);
        if nargin==4 , [par(:,1),par(:,2)] = ind2depo(order,par); end
        varargout{1} = par; 

    case 'ntnode'
        tn = get(t,'tn');
        varargout{1} = length(tn);

    otherwise
        errargt(mfilename,'unknown option','msg'); error('*');
end


%-------------------------------------------------------------%
function tab = ascendants(nodes,order,level,flagdp)
%ASCTABLE Construction of ascendants table.

% Construction of ascendants table.
%----------------------------------
tab = tabofasc(nodes,order,level);

%----------------------------------------%
% Now we may elimate duplicate indices
% And invalid indices (ind<0)
%----------------------------------------%
tab = unique(tab(:));
tab = tab(tab>=0);
if nargin==4 & flagdp
    [tab(:,1),tab(:,2)] = ind2depo(order,tab);
end 
%-------------------------------------------------------------%
function tab = descendants(t,node,type,flagdp)
%DESTABLE Construction of descendants table.

[order,depth,tn] = get(t,'order','depth','tn');
node = depo2ind(order,node);

% Construction of ascendants table.
%----------------------------------
tab = tabofasc(tn,order,depth);

[row,col] = find(tab==node);
switch type
   case 'all'    , first = 1; 
   case 'not_tn' , first = 2; row = row(col>2);
end
last = max(col)-1;
tab  = tab(row,first:last);
tab  = tab(:);
tab  = tab(tab>node);
tab  = [node;unique(tab)];
if nargin==4 & flagdp
    [tab(:,1),tab(:,2)] = ind2depo(order,tab);
end
%-------------------------------------------------------------%
